<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class OrderItem extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'order_id',
        'product_id',
        'quantity',
        'product_snapshot',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'order_id' => 'integer',
        'product_id' => 'integer',
        'quantity' => 'integer',
        'product_snapshot' => 'array',
    ];

    /**
     * Get the order that owns the item.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the product for the order item.
     */
    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Create a snapshot of the product details.
     */
    public function createProductSnapshot(): void
    {
        if ($this->product) {
            $this->product_snapshot = [
                'name' => $this->product->name,
                'size' => $this->product->size,
                'barcode' => $this->product->barcode,
                'brand_name' => $this->product->brand->name,
            ];
            $this->save();
        }
    }

    /**
     * Get product details from snapshot or live product.
     */
    public function getProductDetails(): array
    {
        if ($this->product_snapshot) {
            return $this->product_snapshot;
        }

        if ($this->product) {
            return [
                'name' => $this->product->name,
                'size' => $this->product->size,
                'barcode' => $this->product->barcode,
                'brand_name' => $this->product->brand->name ?? 'N/A',
            ];
        }

        return [
            'name' => 'Product Not Found',
            'size' => 'N/A',
            'barcode' => 'N/A',
            'brand_name' => 'N/A',
        ];
    }
}
