<?php

namespace App\Console\Commands;

use App\Models\Brand;
use App\Models\Product;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ImportHaleonCSV extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'import:haleon-csv {file? : Path to CSV file} {--company= : Company ID to assign products to}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import brands and products from Haleon Record Cards CSV';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $filepath = $this->argument('file') ?? storage_path('app/Haleon Record Cards.csv');

        // Check if file exists in root directory if not in storage
        if (!file_exists($filepath)) {
            $filepath = base_path('Haleon Record Cards.csv');
        }

        if (!file_exists($filepath)) {
            $this->error("CSV file not found: {$filepath}");
            return Command::FAILURE;
        }

        $companyId = $this->option('company');
        if ($companyId) {
            $this->info("Importing for Company ID: {$companyId}");
        }

        $this->info("Importing from: {$filepath}");

        try {
            DB::beginTransaction();

            $stats = $this->importCSV($filepath, $companyId);

            DB::commit();

            $this->newLine();
            $this->info("Import completed successfully!");
            $this->table(
                ['Metric', 'Count'],
                [
                    ['Brands Created', $stats['brands_created']],
                    ['Brands Updated', $stats['brands_updated']],
                    ['Products Created', $stats['products_created']],
                    ['Products Updated', $stats['products_updated']],
                    ['Lines Processed', $stats['lines_processed']],
                    ['Lines Skipped', $stats['lines_skipped']],
                ]
            );

            return Command::SUCCESS;

        } catch (\Exception $e) {
            DB::rollBack();
            $this->error("Import failed: " . $e->getMessage());
            $this->error($e->getTraceAsString());
            return Command::FAILURE;
        }
    }

    /**
     * Import CSV file.
     */
    protected function importCSV(string $filepath, ?int $companyId = null): array
    {
        $handle = fopen($filepath, 'r');
        $lineNumber = 0;
        $currentBrand = null;
        $sortOrder = 0;

        $stats = [
            'brands_created' => 0,
            'brands_updated' => 0,
            'products_created' => 0,
            'products_updated' => 0,
            'lines_processed' => 0,
            'lines_skipped' => 0,
        ];

        $progressBar = $this->output->createProgressBar();
        $progressBar->start();

        while (($data = fgetcsv($handle, 1000, ';')) !== false) {
            $lineNumber++;

            // Skip header rows (first 16 lines)
            if ($lineNumber <= 16) {
                $stats['lines_skipped']++;
                continue;
            }

            $productName = trim($data[0] ?? '');
            $size = trim($data[1] ?? '');
            $barcode = trim($data[2] ?? '');

            // Skip empty rows
            if (empty($productName)) {
                $stats['lines_skipped']++;
                continue;
            }

            // Check if this is a brand header (no size and no barcode)
            if (empty($barcode) && empty($size)) {
                // This is a brand header
                $brand = Brand::firstOrCreate(
                    ['name' => $productName],
                    ['active' => true, 'sort_order' => $sortOrder]
                );

                if ($brand->wasRecentlyCreated) {
                    $stats['brands_created']++;
                    $this->line("\n✓ Brand created: {$productName}");
                } else {
                    $stats['brands_updated']++;
                }

                $currentBrand = $brand;
                $sortOrder++;

            } elseif ($currentBrand && !empty($productName)) {
                // This is a product
                $productData = [
                    'brand_id' => $currentBrand->id,
                    'name' => $productName,
                    'size' => $size,
                    'active' => true,
                    'sort_order' => 0,
                ];

                // Add company_id if provided
                if ($companyId) {
                    $productData['company_id'] = $companyId;
                }

                $product = Product::updateOrCreate(
                    ['barcode' => $barcode],
                    $productData
                );

                if ($product->wasRecentlyCreated) {
                    $stats['products_created']++;
                } else {
                    $stats['products_updated']++;
                }
            }

            $stats['lines_processed']++;
            $progressBar->advance();
        }

        $progressBar->finish();
        fclose($handle);

        return $stats;
    }
}
