<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Wholesaler;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Inertia\Response;

class WholesalerController extends Controller
{
    /**
     * Display a listing of wholesalers.
     */
    public function index(Request $request): Response
    {
        $wholesalers = Wholesaler::query()
            ->withCount(['pharmacies'])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%");
                });
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        return Inertia::render('Admin/Wholesalers/Index', [
            'wholesalers' => $wholesalers,
            'filters' => $request->only(['search']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Wholesalers', 'url' => null],
            ],
        ]);
    }

    /**
     * Show the form for creating a new wholesaler.
     */
    public function create(): Response
    {
        return Inertia::render('Admin/Wholesalers/Create', [
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Wholesalers', 'url' => route('admin.wholesalers.index')],
                ['label' => 'Create', 'url' => null],
            ],
        ]);
    }

    /**
     * Store a newly created wholesaler in storage.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:wholesalers',
            'email' => 'required|email|max:255|unique:wholesalers',
            'phone' => 'nullable|string|max:50',
            'address' => 'nullable|string|max:500',
        ]);

        $wholesaler = Wholesaler::create($validated);

        return redirect()
            ->route('admin.wholesalers.index')
            ->with('success', 'Wholesaler created successfully.');
    }

    /**
     * Show the form for editing the specified wholesaler.
     */
    public function edit(Wholesaler $wholesaler): Response
    {
        return Inertia::render('Admin/Wholesalers/Edit', [
            'wholesaler' => $wholesaler,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
                ['label' => 'Wholesalers', 'url' => route('admin.wholesalers.index')],
                ['label' => $wholesaler->name, 'url' => null],
                ['label' => 'Edit', 'url' => null],
            ],
        ]);
    }

    /**
     * Update the specified wholesaler in storage.
     */
    public function update(Request $request, Wholesaler $wholesaler): RedirectResponse
    {
        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('wholesalers')->ignore($wholesaler->id),
            ],
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('wholesalers')->ignore($wholesaler->id),
            ],
            'phone' => 'nullable|string|max:50',
            'address' => 'nullable|string|max:500',
        ]);

        $wholesaler->update($validated);

        return redirect()
            ->route('admin.wholesalers.index')
            ->with('success', 'Wholesaler updated successfully.');
    }

    /**
     * Remove the specified wholesaler from storage.
     */
    public function destroy(Wholesaler $wholesaler): RedirectResponse
    {
        // Check if wholesaler has pharmacies
        if ($wholesaler->pharmacies()->count() > 0) {
            return redirect()
                ->back()
                ->with('error', 'Cannot delete wholesaler with associated pharmacies.');
        }

        $wholesaler->delete();

        return redirect()
            ->route('admin.wholesalers.index')
            ->with('success', 'Wholesaler deleted successfully.');
    }
}
