<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StorePharmacyRequest;
use App\Http\Requests\UpdatePharmacyRequest;
use App\Http\Resources\PharmacyResource;
use App\Models\Pharmacy;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class PharmacyController extends Controller
{
    /**
     * Display a listing of pharmacies for the authenticated rep's company.
     */
    public function index(Request $request): AnonymousResourceCollection
    {
        // Authorize: check if user can view pharmacies
        $this->authorize('viewAny', Pharmacy::class);

        $query = Pharmacy::query()->with(['createdByRep', 'wholesalers']);

        // Search by pharmacy name, contact person, or phone
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('pharmacy_name', 'like', "%{$search}%")
                  ->orWhere('contact_person', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by created_by_rep_id
        if ($request->has('rep_id')) {
            $query->where('created_by_rep_id', $request->rep_id);
        }

        // Sorting
        $sortField = $request->input('sort', 'pharmacy_name');
        $sortDirection = $request->input('direction', 'asc');
        $query->orderBy($sortField, $sortDirection);

        // Pagination
        $perPage = $request->input('per_page', 15);
        $pharmacies = $query->paginate($perPage);

        return PharmacyResource::collection($pharmacies);
    }

    /**
     * Store a newly created pharmacy.
     */
    public function store(StorePharmacyRequest $request): JsonResponse
    {
        // Authorize: check if user can create pharmacies
        $this->authorize('create', Pharmacy::class);

        // Auto-assign company_id and created_by_rep_id from authenticated user
        $data = $request->validated();
        $data['company_id'] = auth()->user()->company_id;
        $data['created_by_rep_id'] = auth()->id();

        $pharmacy = Pharmacy::create($data);
        $pharmacy->load(['createdByRep', 'wholesalers']);

        return response()->json([
            'success' => true,
            'message' => 'Pharmacy created successfully.',
            'data' => new PharmacyResource($pharmacy),
        ], 201);
    }

    /**
     * Display the specified pharmacy.
     */
    public function show(Pharmacy $pharmacy): JsonResponse
    {
        // Authorize: check if user can view this pharmacy (company-scoped)
        $this->authorize('view', $pharmacy);

        // The global scope will automatically ensure the pharmacy belongs to the user's company
        $pharmacy->load(['createdByRep', 'wholesalers']);

        return response()->json([
            'success' => true,
            'data' => new PharmacyResource($pharmacy),
        ]);
    }

    /**
     * Update the specified pharmacy.
     */
    public function update(UpdatePharmacyRequest $request, Pharmacy $pharmacy): JsonResponse
    {
        // Authorize: check if user can update this pharmacy (company-scoped)
        $this->authorize('update', $pharmacy);

        // The global scope will automatically ensure the pharmacy belongs to the user's company
        $pharmacy->update($request->validated());
        $pharmacy->load(['createdByRep', 'wholesalers']);

        return response()->json([
            'success' => true,
            'message' => 'Pharmacy updated successfully.',
            'data' => new PharmacyResource($pharmacy),
        ]);
    }

    /**
     * Remove the specified pharmacy (soft delete).
     */
    public function destroy(Pharmacy $pharmacy): JsonResponse
    {
        // Authorize: only company_admin can delete pharmacies
        $this->authorize('delete', $pharmacy);

        // The global scope will automatically ensure the pharmacy belongs to the user's company
        $pharmacy->delete();

        return response()->json([
            'success' => true,
            'message' => 'Pharmacy deleted successfully.',
        ]);
    }
}
