<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StorePharmacyWholesalerRequest;
use App\Http\Requests\UpdatePharmacyWholesalerRequest;
use App\Models\Pharmacy;
use App\Models\PharmacyWholesalerAccount;
use Illuminate\Http\JsonResponse;

class PharmacyWholesalerController extends Controller
{
    /**
     * Link pharmacy to wholesaler with account number.
     */
    public function store(StorePharmacyWholesalerRequest $request): JsonResponse
    {
        $validated = $request->validated();

        // Validate that pharmacy belongs to user's company
        $pharmacy = Pharmacy::findOrFail($validated['pharmacy_id']);

        // Check if this pharmacy-wholesaler relationship already exists
        $existingAccount = PharmacyWholesalerAccount::where('pharmacy_id', $validated['pharmacy_id'])
            ->where('wholesaler_id', $validated['wholesaler_id'])
            ->first();

        if ($existingAccount) {
            return response()->json([
                'success' => false,
                'message' => 'This pharmacy is already linked to this wholesaler.',
            ], 422);
        }

        // If is_primary is true, unset any existing primary wholesaler for this pharmacy
        if ($validated['is_primary'] ?? false) {
            PharmacyWholesalerAccount::where('pharmacy_id', $validated['pharmacy_id'])
                ->update(['is_primary' => false]);
        }

        // Create the pharmacy-wholesaler account
        $account = PharmacyWholesalerAccount::create($validated);

        // Load relationships
        $account->load(['pharmacy', 'wholesaler']);

        return response()->json([
            'success' => true,
            'message' => 'Wholesaler account linked successfully.',
            'data' => [
                'id' => $account->id,
                'pharmacy_id' => $account->pharmacy_id,
                'wholesaler_id' => $account->wholesaler_id,
                'account_number' => $account->account_number,
                'is_primary' => $account->is_primary,
                'status' => $account->status,
                'wholesaler' => [
                    'id' => $account->wholesaler->id,
                    'wholesaler_name' => $account->wholesaler->wholesaler_name,
                ],
            ],
        ], 201);
    }

    /**
     * Update pharmacy-wholesaler account details.
     */
    public function update(UpdatePharmacyWholesalerRequest $request, PharmacyWholesalerAccount $pharmacyWholesaler): JsonResponse
    {
        // Validate company ownership through pharmacy
        $pharmacy = Pharmacy::findOrFail($pharmacyWholesaler->pharmacy_id);

        $validated = $request->validated();

        // If is_primary is being set to true, unset any existing primary wholesaler for this pharmacy
        if (($validated['is_primary'] ?? false) && !$pharmacyWholesaler->is_primary) {
            PharmacyWholesalerAccount::where('pharmacy_id', $pharmacyWholesaler->pharmacy_id)
                ->where('id', '!=', $pharmacyWholesaler->id)
                ->update(['is_primary' => false]);
        }

        $pharmacyWholesaler->update($validated);
        $pharmacyWholesaler->load(['pharmacy', 'wholesaler']);

        return response()->json([
            'success' => true,
            'message' => 'Wholesaler account updated successfully.',
            'data' => [
                'id' => $pharmacyWholesaler->id,
                'pharmacy_id' => $pharmacyWholesaler->pharmacy_id,
                'wholesaler_id' => $pharmacyWholesaler->wholesaler_id,
                'account_number' => $pharmacyWholesaler->account_number,
                'is_primary' => $pharmacyWholesaler->is_primary,
                'status' => $pharmacyWholesaler->status,
            ],
        ]);
    }

    /**
     * Remove pharmacy-wholesaler link.
     */
    public function destroy(PharmacyWholesalerAccount $pharmacyWholesaler): JsonResponse
    {
        // Validate company ownership through pharmacy
        $pharmacy = Pharmacy::findOrFail($pharmacyWholesaler->pharmacy_id);

        $pharmacyWholesaler->delete();

        return response()->json([
            'success' => true,
            'message' => 'Wholesaler account link removed successfully.',
        ]);
    }
}
