<?php

namespace App\Http\Controllers\Rep;

use App\Http\Controllers\Controller;
use App\Models\Pharmacy;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class DashboardController extends Controller
{
    /**
     * Display the rep dashboard with list of pharmacies.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Get pharmacies for this company with search
        $query = Pharmacy::where('company_id', $companyId);

        $pharmacies = $query
            ->withCount(['orders'])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('pharmacy_name', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                });
            })
            ->where('status', 'active')
            ->orderBy('pharmacy_name')
            ->paginate(20)
            ->withQueryString();

        // Rep stats
        $stats = [
            'total_pharmacies' => Pharmacy::where('company_id', $companyId)
                ->where('status', 'active')
                ->count(),
            'orders_today' => $user->orders()
                ->whereDate('created_at', today())
                ->count(),
            'orders_this_week' => $user->orders()
                ->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])
                ->count(),
            'orders_this_month' => $user->orders()
                ->whereMonth('created_at', now()->month)
                ->count(),
        ];

        return Inertia::render('Rep/Dashboard', [
            'pharmacies' => $pharmacies,
            'stats' => $stats,
            'filters' => $request->only(['search']),
        ]);
    }
}
