<?php

namespace App\Http\Controllers\SupplierAdmin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Product;
use App\Models\User;
use App\Models\Pharmacy;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class DashboardController extends Controller
{
    /**
     * Display the supplier admin dashboard with statistics and recent activity.
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Ensure user is a company admin
        if (!$user->isCompanyAdmin()) {
            abort(403, 'Unauthorized access');
        }

        // Base queries scoped to supplier's company
        $ordersQuery = Order::where('company_id', $companyId);
        $productsQuery = Product::where('company_id', $companyId);
        $repsQuery = User::where('role', 'rep')->where('company_id', $companyId);
        $pharmaciesQuery = Pharmacy::where('company_id', $companyId);

        // Date ranges
        $today = Carbon::today();
        $weekStart = Carbon::now()->startOfWeek();
        $monthStart = Carbon::now()->startOfMonth();

        // Order statistics
        $stats = [
            'orders_today' => (clone $ordersQuery)->whereDate('created_at', $today)->count(),
            'orders_week' => (clone $ordersQuery)->where('created_at', '>=', $weekStart)->count(),
            'orders_month' => (clone $ordersQuery)->where('created_at', '>=', $monthStart)->count(),
            'total_orders' => (clone $ordersQuery)->count(),

            'active_reps' => (clone $repsQuery)->where('status', 'active')->count(),
            'total_reps' => (clone $repsQuery)->count(),

            'active_products' => (clone $productsQuery)->where('active', true)->count(),
            'total_products' => (clone $productsQuery)->count(),

            'total_pharmacies' => (clone $pharmaciesQuery)->count(),
            'active_pharmacies' => (clone $pharmaciesQuery)->where('status', 'active')->count(),

            'pending_orders' => (clone $ordersQuery)->where('status', 'pending')->count(),
            'processing_orders' => (clone $ordersQuery)->where('status', 'processing')->count(),
            'completed_orders' => (clone $ordersQuery)->where('status', 'completed')->count(),
            'failed_orders' => (clone $ordersQuery)->where('status', 'failed')->count(),
        ];

        // Total items (revenue proxy)
        $revenue = [
            'today' => (clone $ordersQuery)->whereDate('created_at', $today)->sum('total_items'),
            'week' => (clone $ordersQuery)->where('created_at', '>=', $weekStart)->sum('total_items'),
            'month' => (clone $ordersQuery)->where('created_at', '>=', $monthStart)->sum('total_items'),
        ];

        // Recent orders (last 10)
        $recentOrders = (clone $ordersQuery)
            ->with(['pharmacy', 'rep', 'wholesaler'])
            ->latest()
            ->limit(10)
            ->get()
            ->map(function ($order) {
                return [
                    'id' => $order->id,
                    'order_number' => $order->order_number,
                    'pharmacy_name' => $order->pharmacy->pharmacy_name ?? 'N/A',
                    'rep_name' => $order->rep->name ?? 'N/A',
                    'wholesaler_name' => $order->wholesaler->name ?? 'N/A',
                    'total_items' => $order->total_items,
                    'status' => $order->status,
                    'created_at' => $order->created_at->format('Y-m-d H:i:s'),
                ];
            });

        // Top performing reps (by order count this month)
        $topReps = (clone $repsQuery)
            ->withCount(['orders' => function ($query) use ($monthStart) {
                $query->where('created_at', '>=', $monthStart);
            }])
            ->withSum(['orders' => function ($query) use ($monthStart) {
                $query->where('created_at', '>=', $monthStart);
            }], 'total_items')
            ->orderBy('orders_count', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($rep) {
                return [
                    'id' => $rep->id,
                    'name' => $rep->name,
                    'email' => $rep->email,
                    'territory' => $rep->territory,
                    'orders_count' => $rep->orders_count ?? 0,
                    'total_items' => $rep->orders_sum_total_items ?? 0,
                ];
            });

        return Inertia::render('SupplierAdmin/Dashboard', [
            'stats' => $stats,
            'revenue' => $revenue,
            'recentOrders' => $recentOrders,
            'topReps' => $topReps,
            'breadcrumbs' => [
                ['label' => 'Supplier Dashboard', 'url' => null],
            ],
        ]);
    }
}
