<?php

namespace App\Http\Controllers\SupplierAdmin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use App\Models\Pharmacy;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class OrderController extends Controller
{
    /**
     * Display a listing of orders from supplier's reps (read-only).
     */
    public function index(Request $request): Response
    {
        $user = $request->user();
        $companyId = $user->company_id;

        // Get all orders from reps belonging to this supplier's company
        $query = Order::where('company_id', $companyId);

        $orders = $query
            ->with(['pharmacy', 'rep', 'wholesaler', 'items'])
            ->when($request->input('search'), function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('order_number', 'like', "%{$search}%")
                        ->orWhereHas('pharmacy', function ($pharmacyQuery) use ($search) {
                            $pharmacyQuery->where('pharmacy_name', 'like', "%{$search}%");
                        });
                });
            })
            ->when($request->input('status'), function ($q, $status) {
                $q->where('status', $status);
            })
            ->when($request->input('rep_id'), function ($q, $repId) {
                $q->where('rep_id', $repId);
            })
            ->when($request->input('pharmacy_id'), function ($q, $pharmacyId) {
                $q->where('pharmacy_id', $pharmacyId);
            })
            ->when($request->input('date_from'), function ($q, $dateFrom) {
                $q->whereDate('created_at', '>=', $dateFrom);
            })
            ->when($request->input('date_to'), function ($q, $dateTo) {
                $q->whereDate('created_at', '<=', $dateTo);
            })
            ->latest()
            ->paginate(15)
            ->withQueryString();

        // Get filter options scoped to supplier's company
        $reps = User::where('role', 'rep')
            ->where('company_id', $companyId)
            ->orderBy('name')
            ->get(['id', 'name']);

        $pharmacies = Pharmacy::where('company_id', $companyId)
            ->orderBy('pharmacy_name')
            ->get(['id', 'pharmacy_name']);

        return Inertia::render('SupplierAdmin/Orders/Index', [
            'orders' => $orders,
            'reps' => $reps,
            'pharmacies' => $pharmacies,
            'filters' => $request->only(['search', 'status', 'rep_id', 'pharmacy_id', 'date_from', 'date_to']),
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('dashboard')],
                ['label' => 'Orders', 'url' => null],
            ],
        ]);
    }

    /**
     * Display the specified order with all details (read-only).
     */
    public function show(Request $request, Order $order): Response
    {
        $user = $request->user();

        // Ensure order belongs to supplier's company
        if ($order->company_id !== $user->company_id) {
            abort(403, 'Unauthorized action.');
        }

        $order->load([
            'pharmacy',
            'rep',
            'wholesaler',
            'items.product.brand',
            'pdf',
        ]);

        // Group items by brand for display
        $itemsByBrand = $order->items->groupBy(function ($item) {
            return $item->product->brand->name ?? 'Unknown Brand';
        });

        return Inertia::render('SupplierAdmin/Orders/Show', [
            'order' => $order,
            'itemsByBrand' => $itemsByBrand,
            'breadcrumbs' => [
                ['label' => 'Dashboard', 'url' => route('dashboard')],
                ['label' => 'Orders', 'url' => route('supplier-admin.orders.index')],
                ['label' => 'Order #' . $order->order_number, 'url' => null],
            ],
        ]);
    }
}
