<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * EnsureActiveUser Middleware
 *
 * SECURITY: Verifies that the authenticated user has an 'active' status.
 * Inactive users are blocked from accessing the application.
 *
 * Usage:
 * - Apply to all authenticated routes to ensure only active users can access the system
 * - Users with status 'inactive' receive a 403 Forbidden response
 *
 * Example in routes/api.php:
 * Route::middleware(['auth:sanctum', 'ensure.active.user'])->group(function () {
 *     // Protected routes
 * });
 */
class EnsureActiveUser
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Skip check if user is not authenticated (will be handled by auth middleware)
        if (!auth()->check()) {
            return $next($request);
        }

        $user = auth()->user();

        // Verify user has an active status
        if (!property_exists($user, 'status') || $user->status !== 'active') {
            // Log the blocked access attempt
            \Log::warning('Inactive user attempted to access the system', [
                'user_id' => $user->id,
                'email' => $user->email,
                'status' => $user->status ?? 'unknown',
                'ip' => $request->ip(),
                'route' => $request->path(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Your account is currently inactive. Please contact your administrator.',
            ], Response::HTTP_FORBIDDEN);
        }

        return $next($request);
    }
}
