<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Builder;

class Pharmacy extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'company_id',
        'pharmacy_name',
        'registration_number',
        'contact_person',
        'phone',
        'email',
        'physical_address',
        'delivery_address',
        'status',
        'created_by_rep_id',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'company_id' => 'integer',
        'created_by_rep_id' => 'integer',
    ];

    /**
     * The "booted" method of the model.
     */
    protected static function booted(): void
    {
        // Global scope for company isolation - DISABLED until migrations are run
    }

    /**
     * Get the company that owns the pharmacy.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the rep that created the pharmacy.
     */
    public function createdByRep(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by_rep_id');
    }

    /**
     * Get the wholesalers associated with the pharmacy.
     */
    public function wholesalers(): BelongsToMany
    {
        return $this->belongsToMany(Wholesaler::class, 'pharmacy_wholesaler_accounts')
            ->using(PharmacyWholesalerAccount::class)
            ->withPivot(['account_number', 'is_primary', 'status'])
            ->withTimestamps();
    }

    /**
     * Get the orders for the pharmacy.
     */
    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    /**
     * Scope a query to only include active pharmacies.
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }
}
