<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Company;

/**
 * CompanyPolicy
 *
 * Authorization policy for Company model.
 * Enforces strict access control to ensure data isolation.
 *
 * Authorization Rules:
 * - super_admin: Can perform all actions on any company
 * - company_admin: Can view and update their own company only
 * - rep: Can only view their own company
 */
class CompanyPolicy
{
    /**
     * Determine if the user can view any companies.
     *
     * Only super admins can view all companies in the system.
     * Regular users can only view their own company through the view() method.
     *
     * @param User $user
     * @return bool
     */
    public function viewAny(User $user): bool
    {
        return $user->role === 'super_admin';
    }

    /**
     * Determine if the user can view a specific company.
     *
     * Users can only view their own company.
     * Super admins can view any company.
     *
     * @param User $user
     * @param Company $company
     * @return bool
     */
    public function view(User $user, Company $company): bool
    {
        // Super admin can view any company
        if ($user->role === 'super_admin') {
            return true;
        }

        // Users can only view their own company
        return $user->company_id === $company->id;
    }

    /**
     * Determine if the user can create a company.
     *
     * Only super admins can create new companies.
     *
     * @param User $user
     * @return bool
     */
    public function create(User $user): bool
    {
        return $user->role === 'super_admin';
    }

    /**
     * Determine if the user can update a company.
     *
     * Super admins can update any company.
     * Company admins can only update their own company.
     *
     * @param User $user
     * @param Company $company
     * @return bool
     */
    public function update(User $user, Company $company): bool
    {
        // Super admin can update any company
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can only update their own company
        if ($user->role === 'company_admin') {
            return $user->company_id === $company->id;
        }

        // Reps cannot update companies
        return false;
    }

    /**
     * Determine if the user can delete a company.
     *
     * Only super admins can delete companies.
     *
     * @param User $user
     * @param Company $company
     * @return bool
     */
    public function delete(User $user, Company $company): bool
    {
        return $user->role === 'super_admin';
    }

    /**
     * Determine if the user can restore a soft-deleted company.
     *
     * Only super admins can restore companies.
     *
     * @param User $user
     * @param Company $company
     * @return bool
     */
    public function restore(User $user, Company $company): bool
    {
        return $user->role === 'super_admin';
    }

    /**
     * Determine if the user can permanently delete a company.
     *
     * Only super admins can force delete companies.
     *
     * @param User $user
     * @param Company $company
     * @return bool
     */
    public function forceDelete(User $user, Company $company): bool
    {
        return $user->role === 'super_admin';
    }
}
