<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Pharmacy;

/**
 * PharmacyPolicy
 *
 * Authorization policy for Pharmacy model.
 * Enforces multi-tenant data isolation - users can ONLY access pharmacies from their company.
 *
 * Authorization Rules:
 * - super_admin: Can perform all actions on any pharmacy
 * - company_admin: Can manage all pharmacies in their company
 * - rep: Can view and create pharmacies in their company, but cannot delete
 *
 * CRITICAL: All operations must verify company_id matches to prevent cross-company access.
 */
class PharmacyPolicy
{
    /**
     * Determine if the user can view any pharmacies.
     *
     * All authenticated users can view pharmacies from their company.
     * The global scope on Pharmacy model automatically filters by company_id.
     *
     * @param User $user
     * @return bool
     */
    public function viewAny(User $user): bool
    {
        // All authenticated users can view pharmacies (scoped to their company)
        return true;
    }

    /**
     * Determine if the user can view a specific pharmacy.
     *
     * Users can only view pharmacies that belong to their company.
     * CRITICAL: This prevents cross-company data access.
     *
     * @param User $user
     * @param Pharmacy $pharmacy
     * @return bool
     */
    public function view(User $user, Pharmacy $pharmacy): bool
    {
        // Super admin can view any pharmacy
        if ($user->role === 'super_admin') {
            return true;
        }

        // CRITICAL: User can only view if pharmacy belongs to their company
        return $user->company_id === $pharmacy->company_id;
    }

    /**
     * Determine if the user can create a pharmacy.
     *
     * All authenticated users can create pharmacies.
     * The pharmacy will be automatically scoped to their company_id during creation.
     *
     * @param User $user
     * @return bool
     */
    public function create(User $user): bool
    {
        // All authenticated users can create pharmacies (auto-scoped to their company)
        return true;
    }

    /**
     * Determine if the user can update a pharmacy.
     *
     * Users can only update pharmacies that belong to their company.
     *
     * @param User $user
     * @param Pharmacy $pharmacy
     * @return bool
     */
    public function update(User $user, Pharmacy $pharmacy): bool
    {
        // Super admin can update any pharmacy
        if ($user->role === 'super_admin') {
            return true;
        }

        // CRITICAL: User can only update if pharmacy belongs to their company
        return $user->company_id === $pharmacy->company_id;
    }

    /**
     * Determine if the user can delete a pharmacy.
     *
     * Only company admins and super admins can delete pharmacies.
     * Reps cannot delete pharmacies.
     *
     * @param User $user
     * @param Pharmacy $pharmacy
     * @return bool
     */
    public function delete(User $user, Pharmacy $pharmacy): bool
    {
        // Super admin can delete any pharmacy
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can delete pharmacies from their company
        if ($user->role === 'company_admin') {
            return $user->company_id === $pharmacy->company_id;
        }

        // Reps cannot delete pharmacies
        return false;
    }

    /**
     * Determine if the user can restore a soft-deleted pharmacy.
     *
     * Only company admins and super admins can restore pharmacies.
     *
     * @param User $user
     * @param Pharmacy $pharmacy
     * @return bool
     */
    public function restore(User $user, Pharmacy $pharmacy): bool
    {
        // Super admin can restore any pharmacy
        if ($user->role === 'super_admin') {
            return true;
        }

        // Company admin can restore pharmacies from their company
        if ($user->role === 'company_admin') {
            return $user->company_id === $pharmacy->company_id;
        }

        return false;
    }

    /**
     * Determine if the user can permanently delete a pharmacy.
     *
     * Only super admins can force delete pharmacies.
     *
     * @param User $user
     * @param Pharmacy $pharmacy
     * @return bool
     */
    public function forceDelete(User $user, Pharmacy $pharmacy): bool
    {
        return $user->role === 'super_admin';
    }
}
