<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreOrderRequest;
use App\Http\Resources\OrderResource;
use App\Models\Order;
use App\Models\OrderItem;
use App\Jobs\SendOrderEmailJob;
use App\Jobs\GeneratePDFJob;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    /**
     * Display a listing of orders.
     */
    public function index(Request $request): AnonymousResourceCollection
    {
        $query = Order::query()->with(['items.product.brand']);

        // Filter by status
        if ($request->has('status')) {
            $query->status($request->status);
        }

        // Filter by pharmacy
        if ($request->has('pharmacy')) {
            $query->byPharmacy($request->pharmacy);
        }

        // Filter by date range
        if ($request->has('start_date')) {
            $query->dateRange($request->start_date, $request->end_date);
        }

        // Sorting
        $sortField = $request->input('sort', 'created_at');
        $sortDirection = $request->input('direction', 'desc');
        $query->orderBy($sortField, $sortDirection);

        // Pagination
        $perPage = $request->input('per_page', 15);
        $orders = $query->paginate($perPage);

        return OrderResource::collection($orders);
    }

    /**
     * Store a newly created order.
     */
    public function store(StoreOrderRequest $request): JsonResponse
    {
        try {
            DB::beginTransaction();

            // Create order
            $order = Order::create([
                'pharmacy_name' => $request->pharmacy_name,
                'email' => $request->email,
                'notes' => $request->notes,
                'status' => 'pending',
            ]);

            // Create order items
            foreach ($request->items as $item) {
                $orderItem = OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                ]);

                // Create product snapshot for historical accuracy
                $orderItem->createProductSnapshot();
            }

            // Update total items
            $order->updateTotalItems();

            // Dispatch jobs for PDF generation and email sending
            GeneratePDFJob::dispatch($order);
            SendOrderEmailJob::dispatch($order);

            DB::commit();

            // Reload relationships
            $order->load(['items.product.brand', 'pdf']);

            return response()->json([
                'success' => true,
                'message' => 'Order created successfully. Email will be sent shortly.',
                'data' => new OrderResource($order),
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to create order.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified order.
     */
    public function show(Order $order): JsonResponse
    {
        $order->load(['items.product.brand', 'pdf']);

        return response()->json([
            'success' => true,
            'data' => new OrderResource($order),
        ]);
    }

    /**
     * Update the specified order status.
     */
    public function updateStatus(Request $request, Order $order): JsonResponse
    {
        $validated = $request->validate([
            'status' => ['required', 'in:pending,processing,completed,failed'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);

        $order->update($validated);

        if ($validated['status'] === 'completed') {
            $order->markAsCompleted();
        }

        return response()->json([
            'success' => true,
            'message' => 'Order status updated successfully.',
            'data' => new OrderResource($order),
        ]);
    }

    /**
     * Resend order email.
     */
    public function resendEmail(Order $order): JsonResponse
    {
        SendOrderEmailJob::dispatch($order);

        return response()->json([
            'success' => true,
            'message' => 'Order email queued for sending.',
        ]);
    }

    /**
     * Regenerate order PDF.
     */
    public function regeneratePDF(Order $order): JsonResponse
    {
        GeneratePDFJob::dispatch($order);

        return response()->json([
            'success' => true,
            'message' => 'PDF generation queued.',
        ]);
    }
}
